#!/bin/bash

##############################
## MAIL SERVER INSTALLATION ##
##############################
# Author: Lukas Feiler <lukas.feiler@lukasfeiler.com>


### PREREQUISTES ###
# - make sure the following packages are installed:
#    gcc
#    glibc-devel
#    openssl
#    openssl-devel 
#    make
#    mutt
# - make sure that DNS works;
#   the servers FQDN should resolve to the right IP and that IP
#   should reverse resolve to the servers FQDN
# - the /var partition must allow the use of SUID/SGID executables;
#   symlink /var/qmail to some other partition if it doesn't


export HOME_DIR=`(cd $(dirname $0); pwd)`
. $HOME_DIR/common.sh || exit 1

printUsage()
{
cat >&1 <<END
Usage:
 $0 [-C] [-U]
 -C  remove extraced files
 -U  uninstall *everything*; know what you are doing!
END
}

while getopts ":CU" opt; do
    case $opt in
        C ) cleanup=true;;
        U ) uninstall=true;;
        ? ) printUsage
            exit 1 ;;
    esac
done

if [ "$cleanup" = "true" ] ; then
    echo "Removing $EXTRACT_DIR/* ..."
    rm -rf $EXTRACT_DIR/*
fi

if [ "$uninstall" = "true" ] ; then
    echo "Running qmailctl stop ..."
    qmailctl stop
    echo "Removing inittab entry ..."
    grep -v svscanboot /etc/inittab > /tmp/inittab.tmp || exit
    cat /tmp/inittab.tmp > /etc/inittab
    rm -f /tmp/inittab.tmp
    echo "Running telinit q ..."
    telinit q
    sleep 3
    echo "Killing all svscanboot processes ..."
    killall svscanboot
    sleep 3
    echo "Killing all supervise processes ..."
    killall supervise
    sleep 3
    echo "Killing all svscan processes ..."
    killall svscan
    sleep 3
    echo "Killing all readproctitle processes ..."
    killall readproctitle
    sleep 3
    echo "Removing qmail users and groups ..."
    userdel alias
    userdel qmaild
    userdel qmaill
    userdel qmailp
    userdel qmailq
    userdel qmailr
    userdel qmails
    userdel clamd
    userdel -r mailadmin
    groupdel qmail
    groupdel nofiles
    groupdel mailadmin
    groupdel clamd
    echo "Removing /var/qmail ..."
    rm -rf /var/qmail
    echo "Removing /var/clamd ..."
    rm -rf /var/clamd
    echo "Removing /package ..."
    rm -rf /package
    echo "Removing /service ..."
    rm -rf /service
    echo "Removing /command ..."
    rm -rf /command
    echo "Removing /etc/tcp.smtp ..."
    rm -f /etc/tcp.smtp
    echo "Removing /usr/local/etc/clamd.conf & freshclam.conf ..."
    rm -f /usr/local/etc/clamd.conf /usr/local/etc/freshclam.conf
    echo "Removing /var/log/qmail ..."
    rm -rf /var/log/qmail
    echo "Removing /var/log/clamd ..."
    rm -rf /var/log/clamd
    echo "Removing /var/log/freshclam ..."
    rm -rf /var/log/freshclam
    echo "Removing /var/log/qmail-qfilter-clamdscan.err ..."
    rm -f /var/log/qmail-qfilter-clamdscan.err
fi

if [ "$uninstall" = "true" -o "$cleanup" = "true" ] ; then
    exit 0
fi

### PREREQUISITES ###
# - basic qmail configuration - make first sure that DNS works;
#   the servers FQDN should resolve to the right IP and that IP
#   should reverse resolve to the servers FQDN
echo "Checking prerequisites:"

echo "Checking DNS ..."
FQDN=`hostname -f` \
 || DNS_ERROR="get hostname"
FQDN_IP=`host $FQDN` \
 || DNS_ERROR="resolve your hostname's IP"
FQDN_IP=`echo $FQDN_IP | tail -1 | awk '{print $4}'`
FQDN_REVERSED=`host $FQDN_IP` \
 || DNS_ERROR="reverse-resolve the obtained IP"
FQDN_REVERSED=`echo $FQDN_REVERSED | tail -1 | awk '{print $5}'`
FQDN_REVERSED=${FQDN_REVERSED%\.}
if [ "$FQDN" = "" -o "$FQDN" != "$FQDN_REVERSED" ] ; then
    echo "ERROR: DNS check failed --> cound not $DNS_ERROR"
    echo " the full qualified domain name (FQDN) returned by 'hostname -f' should resolve"
    echo " to your IP that reverse-resolves to your FQDN"
    echo " --> hostname -f: $FQDN"
    echo " --> host $FQDN: $FQDN_IP"
    echo " --> host $FQDN_IP: $FQDN_REVERSED"
    exit 1
fi


checkRPM glibc
checkRPM glibc-devel
checkRPM openssl
checkRPM openssl-devel
checkRPM patch
checkRPM mutt

echo "Creating directories needed for compilation ..."
mkdir -p $PACKAGES_DIR || exit 1
mkdir -p $EXTRACT_DIR || exit 1


############
# NETQMAIL #
############
# netqmail-1.05 is qmail-1.03 plus some recommended patches
# see http://qmail.org/top.html#patches and http://www.qmail.org/netqmail/

echo "Extracting netqmail ..."
(cd $EXTRACT_DIR && tar xzpf $PACKAGES_DIR/netqmail-$VERSION_NETQMAIL.tar.gz) || exit 1

# patch qmail sources
echo "Applying netqmail patches (watch out for error) ..."
# watch out for errors
(cd $EXTRACT_DIR/netqmail-$VERSION_NETQMAIL && ./collate.sh) || exit 1
echo -n "Hit ENTER to confirm that everything went as expected: "
read in


### TLS & SMTP-AUTH PATCH ###
# sources:
# http://www.fehcom.de/qmail/smtpauth.html  - SMTP Authentication
# http://shupp.org
# NOTE: the patch changes the command line arguments of qmail-smtpd; so it is not
# <domain_name_for_CRAM_MD5> <password_checker> <command> but only
# <password_checker> <command> (no matter if CRAM MD5 is used or not).
# Please see the following thread for details:
# http://msgs.securepoint.com/cgi-bin/get/qmail0402/285.html
echo "Applying shupp.org tls-smtpauth patch ..."

(cd $EXTRACT_DIR/netqmail-$VERSION_NETQMAIL/netqmail-$VERSION_NETQMAIL && \
  patch < $PATCHES_DIR/netqmail-$VERSION_NETQMAIL-tls-smtpauth-20051423.patch) || exit 1


### DOUBLE BOUNCE PATCH ###
# optional: discard bouble bounces directly if /var/qmail/control/doublebounceto
# exists but contains a blank line
echo "Applying double bounce patch ..."
(cd $EXTRACT_DIR/netqmail-$VERSION_NETQMAIL/netqmail-$VERSION_NETQMAIL && \
  patch < $PATCHES_DIR/doublebounce-trim.patch) || exit 1


### CRAM-MD5 DEACTIVATION PATCH ###
# to disable AUTH CRAM-MD5 (which is necessary if we do not have
# all passwords available in plain text) we comment out
# "#define CRAM_MD5" at around line 28 (using /* and */) in qmail-smtpd.c;
# To do so is recommended as we use TLS anyway! Furthermore: most MUAs
# don't support CRAM-MD5 anyway.

echo "Applying CRAM-MD5 deactivation patch ..."
(cd $EXTRACT_DIR/netqmail-$VERSION_NETQMAIL/netqmail-$VERSION_NETQMAIL && \
  patch < $PATCHES_DIR/qmail-smtpd_no_CRAM_MD5.patch) || exit 1

echo "Creating /var/qmail ..."
mkdir -p /var/qmail || exit 1


echo "Creating required users and groups..."
# create required users and groups if not already
# existing (as described in netqmail-$VERSION_NETQMAIL/netqmail-$VERSION_NETQMAIL/INSTALL.ids)
# Note: the group nofiles should never own any files; run
#       ``find / -group nofiles'' to verify that.
# Note: the -r option is non-standard but will work on SUSE and Fedora;
#       the created account will be treaed as system account.
groupadd -r nofiles || exit 1
useradd -r -g nofiles -d /var/qmail/alias alias || exit 1
useradd -r -g nofiles -d /var/qmail qmaild || exit 1
useradd -r -g nofiles -d /var/qmail qmaill || exit 1
useradd -r -g nofiles -d /var/qmail qmailp || exit 1
groupadd -r qmail || exit 1
useradd -r -g qmail -d /var/qmail qmailq || exit 1
useradd -r -g qmail -d /var/qmail qmailr || exit 1
useradd -r -g qmail -d /var/qmail qmails || exit 1

echo "Running 'make' for qmail ..."
(cd $EXTRACT_DIR/netqmail-$VERSION_NETQMAIL/netqmail-$VERSION_NETQMAIL && \
  make) || exit 1

echo "Running 'make setup check' for qmail ..."
(cd $EXTRACT_DIR/netqmail-$VERSION_NETQMAIL/netqmail-$VERSION_NETQMAIL && \
  make setup check) || exit 1

echo "Running 'make cert' for qmail ..."
(cd $EXTRACT_DIR/netqmail-$VERSION_NETQMAIL/netqmail-$VERSION_NETQMAIL && \
  make cert) || exit 1

echo "Running 'make tmprsadh' for qmail ..."
(cd $EXTRACT_DIR/netqmail-$VERSION_NETQMAIL/netqmail-$VERSION_NETQMAIL && \
  make tmprsadh) || exit 1

echo "Running './config' for qmail ..."
(cd $EXTRACT_DIR/netqmail-$VERSION_NETQMAIL/netqmail-$VERSION_NETQMAIL && \
  ./config) || exit 1

echo -n "I hope this is fine... Hit ENTER to continue:"
read in


echo "Installing qmail man pages ..."
mkdir -p /usr/local/man/man1/ || exit 1
mkdir -p /usr/local/man/man5/ || exit 1
mkdir -p /usr/local/man/man7/ || exit 1
mkdir -p /usr/local/man/man8/ || exit 1

cp /var/qmail/man/man1/* /usr/local/man/man1/ || exit 1
cp /var/qmail/man/man5/* /usr/local/man/man5/ || exit 1
cp /var/qmail/man/man7/* /usr/local/man/man7/ || exit 1
cp /var/qmail/man/man8/* /usr/local/man/man8/ || exit 1



###############
# DAEMONTOOLS #
###############
# tools for managing UNIX services; we will use it to run:
# - qmail-smtpd
# - qmail-send
# - clamd
# - frechclam
# - spamd


echo "STARTING DAEMONTOOLS INSTALLATION"

echo "mkdir -p /package ..."
mkdir -p /package || exit 1
echo "chmod 1755 /package ..."
chmod 1755 /package || exit 1

echo "Extracting daemontools package ..."
(cd /package && tar xzpf $PACKAGES_DIR/daemontools-$VERSION_DAEMONTOOLS.tar.gz) || exit 1

# apply the errno patch required for GNU glibc > 2.3.1;
# ``rpm -qa | grep glibc'' would tell the version
echo "Applying errno patch to daemontools ..."
(cd /package/admin/daemontools-$VERSION_DAEMONTOOLS/src/ && \
  patch < $EXTRACT_DIR/netqmail-$VERSION_NETQMAIL/other-patches/daemontools-$VERSION_DAEMONTOOLS.errno.patch) || exit 1

# do the actual installation; this will also add a line to /etc/inittab
echo "Running install for daemontools ..."
(cd /package/admin/daemontools-$VERSION_DAEMONTOOLS/ && \
 ./package/install) || exit 1

echo "Checking that svscan is running ..."
sleep 2
SVSCAN_PID=`pidof svscan`
if [ "$SVSCAN_PID" = "" ] ; then
    echo "ERROR svscan is not running; check /etc/inittab"
    exit 1
fi

echo "Extracting daemontools man pages ..."
(cd $EXTRACT_DIR && tar xzpf $PACKAGES_DIR/daemontools-$VERSION_DAEMONTOOLS-man.tar.gz) || exit 1

# install man pages in /usr/local/man
echo "Installing daemontools man pages ..."
chown -R root:root $EXTRACT_DIR/daemontools-man || exit 1
chmod 644 $EXTRACT_DIR/daemontools-man/* || exit 1
mkdir -p /usr/local/man/man8/ || exit 1
cp $EXTRACT_DIR/daemontools-man/*.8 /usr/local/man/man8/ || exit 1


#############
# UCSPI-TCP #
#############
# ucspi-tcp contains tcpserver which must be used instead of xinetd

echo "STARTING UCSPI-TCP INSTALLATION ..."
echo "Extracting ucspi-tcp ..."
(cd $EXTRACT_DIR && tar xzpf $PACKAGES_DIR/ucspi-tcp-$VERSION_UCSPI_TCP.tar.gz) || exit 1

# apply the errno patch required for GNU glibc > 2.3.1;
# ``rpm -qa | grep glibc'' would tell the version
echo "Applying errno patch to ucspi-tcp ..."
(cd $EXTRACT_DIR/ucspi-tcp-$VERSION_UCSPI_TCP && \
  patch < $EXTRACT_DIR/netqmail-$VERSION_NETQMAIL/other-patches/ucspi-tcp-$VERSION_UCSPI_TCP.errno.patch) || exit 1

echo "Running make for ucspi-tcp ..."
(cd $EXTRACT_DIR/ucspi-tcp-$VERSION_UCSPI_TCP && make) || exit 1

echo "Running 'make setup check' for ucspi-tcp ..."
(cd $EXTRACT_DIR/ucspi-tcp-$VERSION_UCSPI_TCP && make setup check) || exit 1

# MAN PAGES
mkdir -p /usr/local/man/man1 || exit 1

echo "Extracting ucspi-tcp man pages ..."
(cd $EXTRACT_DIR && tar xzf $PACKAGES_DIR/ucspi-tcp-$VERSION_UCSPI_TCP-man.tar.gz) || exit 1

echo "Changing ownership and permissions of ucspi-tcp man pages ..."
(cd $EXTRACT_DIR && chown -R root:root ucspi-tcp-$VERSION_UCSPI_TCP-man) || exit 1
(cd $EXTRACT_DIR && chmod 644 ucspi-tcp-$VERSION_UCSPI_TCP-man/*) || exit 1

echo "Installing ucspi-tcp man pages in /usr/local/man ..."
cp $EXTRACT_DIR/ucspi-tcp-$VERSION_UCSPI_TCP-man/*.1 /usr/local/man/man1/ || exit 1


########################
# MAILADMIN USER SETUP #
########################
echo "SETTING UP MAILADMIN ACCOUNT ..."

# add Maildir to skel
echo "Creating /etc/skel/Maildir ..."
rm -rf /etc/skel/Maildir
/var/qmail/bin/maildirmake /etc/skel/Maildir || exit 1

# set the MAILDIR and the MAIL variable to ~/Maildir so that local
# shell users will not have to specify mutt's -f option
echo "Setting the variables MAILDIR and MAIL to '~/Maildir' ..."
echo "MAILDIR=~/Maildir" >> /etc/profile.d/paths.sh || exit 1
echo "MAIL=~/Maildir" >> /etc/profile.d/paths.sh || exit 1

# add the mail server administrator account and put him in his own group
echo "Creating system group mailadmin ..."
groupadd mailadmin
echo "Creating system account mailadmin ..."
useradd -g mailadmin -m -k /etc/skel mailadmin

# mailadmin should have a Maildir now:
echo "Checking for mailadmin's Maildir ..."
if [ ! -d /home/mailadmin/Maildir ] ; then
    echo "ERROR: /home/mailadmin/Maildir does not exist!"
    exit 1
fi

# we do not want mailadmin's mails to be spam-filtered
echo "Writing './Maildir/' to mailadmin's .qmail file ..."
echo "./Maildir/" > /home/mailadmin/.qmail || exit 1

echo "Changing owner and permissions of mailadmin's .qmail file ..."
chown mailadmin:mailadmin /home/mailadmin/.qmail || exit 1
chmod 600 /home/mailadmin/.qmail || exit 1


#################
# CHECKPASSWORD #
#################
# checkpassword is qmail's standard implementation for verifying user credentials
# we will use courierpasswd instead but checkpassword might be handy for debugging

echo "Extracting checkpassword ..."
(cd $EXTRACT_DIR && tar xzf $PACKAGES_DIR/checkpassword-$VERSION_CHECKPASSWORD.tar.gz) || exit 1

echo "Applying errno patch to checkpassword ..."
(cd $EXTRACT_DIR/checkpassword-$VERSION_CHECKPASSWORD && \
  patch < $EXTRACT_DIR/netqmail-$VERSION_NETQMAIL/other-patches/checkpassword-$VERSION_CHECKPASSWORD.errno.patch) || exit 1

echo "Running make for checkpassword ..."
(cd $EXTRACT_DIR/checkpassword-$VERSION_CHECKPASSWORD && make) || exit 1

echo "Running 'make setup check' for checkpassword ..."
(cd $EXTRACT_DIR/checkpassword-$VERSION_CHECKPASSWORD && make setup check) || exit 1



#######################
# QMAIL CONFIGURATION #
#######################

# create /var/qmail/rc
# See http://www.lifewithqmail.org/lwq.html#installation, Section "2.8.1. /var/qmail/rc"
echo "Creating /var/qmail/rc ..."
cp $SCRIPTS_DIR/var_qmail_rc /var/qmail/rc || exit 1
chmod 755 /var/qmail/rc || exit 1


# to set the default delivery method to Maildirs
# defaultdelivery is a non-standard configuration file used by our custom /var/qmail/rc script
# THIS IS ONLY FOR NOW - WE WILL SET THIS LATER TO USE maildrop
echo "Setting defaultdelivery ..."
echo "./Maildir/" > /var/qmail/control/defaultdelivery || exit 1

# get the startup/shutdown script from lifewithqmail.org
# and move it to /var/qmail/bin/qmailctl
# See http://www.lifewithqmail.org/lwq.html#installation, Section "2.8.2.1. The qmailctl script"
# alternatively you can get it at http://www.lifewithqmail.org/qmailctl-script-dt70
echo "Creating /var/qmail/bin/qmailctrl ..."
cp $SCRIPTS_DIR/var_qmail_bin_qmailctl /var/qmail/bin/qmailctl || exit 1
chmod 755 /var/qmail/bin/qmailctl || exit 1
rm -f /usr/bin/qmailctl
ln -s /var/qmail/bin/qmailctl /usr/bin || exit 1


# creation of the supervise directories for the qmail services
echo "Creating supervise directories for qmail services ..."
mkdir -p /var/qmail/supervise/qmail-send/log || exit 1
mkdir -p /var/qmail/supervise/qmail-smtpd/log || exit 1

# create /var/qmail/supervise/qmail-send/run
echo "Creating /var/qmail/supervise/qmail-send/run ..."
cp $SCRIPTS_DIR/var_qmail_supervise_qmail-send_run \
  /var/qmail/supervise/qmail-send/run || exit 1

# create /var/qmail/supervise/qmail-send/log/run
echo "Creating /var/qmail/supervise/qmail-send/log/run ..."
cp $SCRIPTS_DIR/var_qmail_supervise_qmail-send_log_run \
  /var/qmail/supervise/qmail-send/log/run || exit 1

# create /var/qmail/supervise/qmail-smtpd/run
echo "Creating /var/qmail/supervise/qmail-smtpd/run ..."
cp $SCRIPTS_DIR/var_qmail_supervise_qmail-smtpd_run \
  /var/qmail/supervise/qmail-smtpd/run || exit 1

# create /var/qmail/supervise/qmail-smtpd/log/run
echo "Creating /var/qmail/supervise/qmail-smtpd/log/run ..."
cp $SCRIPTS_DIR/var_qmail_supervise_qmail-smtpd_log_run \
  /var/qmail/supervise/qmail-smtpd/log/run || exit 1

# to discard double bounces create doublebounceto with an empty line
# usually only spam double bounces
echo $DOUBLE_BOUNCE > /var/qmail/control/doublebounceto || exit 1

# set the "momory" limit of the tcpserver starting qmail-smtpd
# the limit is set in /var/qmail/supervise/qmail-smtpd/run using
# /usr/local/bin/softlimit -m; see its man page for details
# this is a non-standard configuration file used by our custom
# /var/qmail/supervise/qmail-smtpd/run script; we use a very high value here
echo "Setting /var/qmail/control/smtpdmemorylimit to $SMTP_MEMORY_LIMIT ..."
echo $SMTP_MEMORY_LIMIT > /var/qmail/control/smtpdmemorylimit || exit 1
chmod 644 /var/qmail/control/smtpdmemorylimit || exit 1

# to set the number of concurrent incoming connections
# concurrencyincoming is a non-standard configuration file used
# by our custom /var/qmail/supervise/qmail-smtpd/run script
echo "Setting /var/qmail/control/concurrencyincoming to $CONCURRENCY_INCOMING ..."
echo $CONCURRENCY_INCOMING > /var/qmail/control/concurrencyincoming || exit 1
chmod 644 /var/qmail/control/concurrencyincoming || exit 1

# to set the number of remote deliveries allowed at once
echo "Setting /var/qmail/control/concurrencyremote to $CONCURRENCY_REMOTE ..."
echo $CONCURRENCY_REMOTE > /var/qmail/control/concurrencyremote || exit 1
chmod 644 /var/qmail/control/concurrencyremote || exit 1

# to set the number of local deliveries allowed at once
echo "Setting /var/qmail/control/concurrencylocal to $CONCURRENCY_LOCAL ..."
echo $CONCURRENCY_LOCAL > /var/qmail/control/concurrencylocal || exit 1
chmod 644 /var/qmail/control/concurrencylocal || exit 1

# set the maximum size in bytes of a message received via SMTP
# This is the default and can be overwritten with the DATABYTES variable
# in /etc/tcp.smtp.
echo "Setting /var/qmail/control/databytes to $DATABYTES ..."
echo $DATABYTES > /var/qmail/control/databytes || exit 1
chmod 644 /var/qmail/control/databytes

# Set the maximum lifetime (in seconds) of a message in the queue
# after this time period temporary errors are considered permanent and
# the message is bounced
echo "Setting /var/qmail/control/queuelifetime to $QUEUELIFETIME ..."
echo $QUEUELIFETIME > /var/qmail/control/queuelifetime || exit 1
chmod 644 /var/qmail/control/queuelifetime || exit 1

# how long to wait for each response of a remote SMTP
# server before giving up; this is essential because the default is 1200 secs
echo "Setting /var/qmail/control/timeoutremote to $TIMEOUTREMOTE ..."
echo $TIMEOUTREMOTE > /var/qmail/control/timeoutremote || exit 1
chmod 644 /var/qmail/control/timeoutremote || exit 1

# how long to wait for each response of a remote SMTP
# client before giving up; this is essential too, because the default is 1200 secs
echo "Setting /var/qmail/control/timeoutsmtpd to $TIMEOUTSMTPD ..."
echo $TIMEOUTSMTPD > /var/qmail/control/timeoutsmtpd || exit 1
chmod 644 /var/qmail/control/timeoutsmtpd || exit 1

# set the maximum multilog log file size in bytes; if the logfile is
# bigger than that it will be rotated
# see http://cr.yp.to/daemontools/multilog.html
# This a non-standard control file; it is used by
# /var/qmail/supervise/qmail-smtpd/log/run and /var/qmail/supervise/qmail-send/log/run
echo "Setting /var/qmail/control/multilogsize to $QMAIL_MULTILOGSIZE ..."
echo $QMAIL_MULTILOGSIZE > /var/qmail/control/multilogsize || exit 1
chmod 644 /var/qmail/control/multilogsize || exit 1

# set the maximum number of multilog log files to keep; old log files will be deleted
# see http://cr.yp.to/daemontools/multilog.html
# This a non-standard control file; it is used by
# /var/qmail/supervise/qmail-smtpd/log/run and /var/qmail/supervise/qmail-send/log/run
echo "Setting /var/qmail/control/multilognum to $QMAIL_MULTILOGNUM ..."
echo $QMAIL_MULTILOGNUM > /var/qmail/control/multilognum || exit 1
chmod 644 /var/qmail/control/multilognum || exit 1

echo -n "Please verify that the values written to the control files are OK:"
read in

# to make all the run files executable
echo "Setting permissions for qmail supervise scripts ..."
chmod 755 /var/qmail/supervise/qmail-send/run || exit 1
chmod 755 /var/qmail/supervise/qmail-send/log/run || exit 1
chmod 755 /var/qmail/supervise/qmail-smtpd/run || exit 1
chmod 755 /var/qmail/supervise/qmail-smtpd/log/run || exit 1

# Setup log directories
echo "Creating logging directory /var/log/qmail/smtpd ..."
mkdir -p /var/log/qmail/smtpd || exit 1
chown qmaill /var/log/qmail /var/log/qmail/smtpd || exit 1

# link the supervise directories into /service
echo "Linking the supervise directories into /service ..."
ln -s /var/qmail/supervise/qmail-send /var/qmail/supervise/qmail-smtpd /service || exit 1

# Allow the local host to inject mail via SMTP:
echo "Adding entry to /etc/tcp.smtp for localhost ..."
echo '127.:allow,RELAYCLIENT=""' >>/etc/tcp.smtp || exit 1

# will call /usr/local/bin/tcprules
echo "Running qmailctl cdb ..."
qmailctl cdb || exit 1

# stop qmail that is running by now (was automatically started after creating the links in /service)
echo "Running qmailctl stop ..."
qmailctl stop

sleep 1
# make sure no other process is listening on port 25
if [ "`netstat -ap | grep smtp`" != "" ] ; then
    echo "It seems that there already is an SMTP server running on port 25."
    echo -n "Shut it down and hit ENTER when done:"
    read in;
fi

# optional: if sendmail is installed:
mv /usr/lib/sendmail /usr/lib/sendmail.old 2>/dev/null           # ignore errors
mv /usr/sbin/sendmail /usr/sbin/sendmail.old 2>/dev/null         # ignore errors


# create the sendmail links
ln -s /var/qmail/bin/sendmail /usr/lib
ln -s /var/qmail/bin/sendmail /usr/sbin

# Create System Aliases
#
# postmaster    - required by RFC2821; points to the mail adminstrator
# mailer-daemon - de facto standard recipient for some bounces
# root          - mail for the root account must be sent to some other user
#                 qmail won't deliver mail to the root user for security reasons
echo "Creating aliases for postmaster, mailer-daemon and root ..."
echo $ROOT_MAILS_TO > /var/qmail/alias/.qmail-root || exit 1
echo mailadmin > /var/qmail/alias/.qmail-postmaster || exit 1
echo mailadmin > /var/qmail/alias/.qmail-mailer-daemon || exit 1
chmod 600 /var/qmail/alias/.qmail-root \
          /var/qmail/alias/.qmail-postmaster \
          /var/qmail/alias/.qmail-mailer-daemon || exit 1
chown alias:nofiles /var/qmail/alias/.qmail-root \
                    /var/qmail/alias/.qmail-postmaster \
                    /var/qmail/alias/.qmail-mailer-daemon || exit 1

echo "Running telinit q (and waiting 5 seconds) ..."
telinit q
sleep 5

# now start qmail
echo "Starting qmail by running 'qmailctl start' ..."
qmailctl start || exit 1

echo "Wating 10 seconds for qmail to start ..."
sleep 10

# verify successful startup
echo "Please verify successful startup (output of 'qmailctl stat' follows):"
qmailctl stat
echo -n "Hit ENTER to confirm a successful startup:"
read in;

# check your installation by running 
# (the original version is available at http://lifewithqmail.org/inst_check)
echo "Checking our qmail installation by running lifewithqmail.org's inst_check ..."
sh $SCRIPTS_DIR/inst_check

echo "WE WILL RUN SOME TESTS NOW:"
echo ""
echo "Sending a mail to root (will be delivered to $ROOT_MAILS_TO):"
$TESTS_DIR/inject_mail.sh root
echo ""
echo "The following we expect to fail:"
$TESTS_DIR/inject_mail.sh no_such_user

echo -n "Use the MUA mutt to verify the deliveries and hit ENTER when done:"
read in

echo ""
echo "CONGRATULATIONS! The first part of your qmail installation was successfully completed."
