#!/bin/bash

export HOME_DIR=`(cd $(dirname $0); pwd)`
. $HOME_DIR/common.sh || exit 1

printUsage()
{
cat >&1 <<END
Usage:
 $0 [-C] [-U]
 -C  remove extraced files
 -U  uninstall *everything*; know what you are doing!
END
}

while getopts ":CU" opt; do
    case $opt in
        C ) cleanup=true;;
        U ) uninstall=true;;
        ? ) printUsage
            exit 1 ;;
    esac
done

if [ "$cleanup" = "true" ] ; then
    echo "Removing $EXTRACT_DIR/* ..."
    rm -rf $EXTRACT_DIR/*
fi

if [ "$uninstall" = "true" ] ; then
    echo "Running clamdctl stop ..."
    clamdctl stop
    echo "Running freshclamctl stop ..."
    freshclamctl stop
    sleep 3
    echo "Removing clamd user and group ..."
    userdel clamd
    groupdel clamd
    echo "Removing /var/clamd ..."
    rm -rf /var/clamd
    echo "Removing /service/clamd ..."
    rm -rf /service/clamd
    echo "Removing /service/freshclam ..."
    rm -rf /service/freshclam
    echo "Removing /usr/local/etc/clamd.conf & freshclam.conf ..."
    rm -f /usr/local/etc/clamd.conf /usr/local/etc/freshclam.conf
    echo "Removing /var/log/clamd ..."
    rm -rf /var/log/clamd
    echo "Removing /var/log/freshclam ..."
    rm -rf /var/log/freshclam
    echo "Removing /var/log/qmail-qfilter-clamdscan.err ..."
    rm -f /var/log/qmail-qfilter-clamdscan.err
fi

if [ "$uninstall" = "true" -o "$cleanup" = "true" ] ; then
    exit 0
fi


### PREREQUISITES ###
# - make sure the following packages are installed:
#    zlib
#    zlib-devel
#    bzip2
#    bzip2-devel (on SUSE it's part of bzip2)
echo "Checking prerequisites:"

checkRPM zlib
checkRPM zlib-devel
checkRPM bzip2
#checkRPM bzip2-devel
checkRPM curl
checkRPM curl-devel
checkRPM libidn
checkRPM libidn-devel
checkRPM gmp

######################
# INSTALLING CLAM AV #
######################
echo "STARTING CLAM AV INSTALLATION"

# install GMP for signature validation during virus database updates
# http://ftp.gnu.org/gnu/gmp/gmp-4.1.4.tar.gz
#echo "Extracting gmp (needed for signature validation) ..."
#(cd $EXTRACT_DIR && tar zxf $PACKAGES_DIR/gmp-4.1.4.tar.gz) || exit 1
#
#echo "Running ./configure for gmp ..."
#(cd $EXTRACT_DIR/gmp-4.1.4 && ./configure) || exit 1
#
#echo "Running make for gmp ..."
#(cd $EXTRACT_DIR/gmp-4.1.4 && make) || exit 1
#
#echo "Running make check for gmp ..."
#(cd $EXTRACT_DIR/gmp-4.1.4 && make check) || exit 1
#
#echo "Running make install for gmp ..."
#(cd $EXTRACT_DIR/gmp-4.1.4 && make install) || exit 1

echo "Extracting Clam AV ..."
(cd $EXTRACT_DIR && tar zxf $PACKAGES_DIR/clamav-$VERSION_CLAMAV.tar.gz) || exit 1

# the -r option is non-standard but works on FC and SUSE
echo "Adding clamd system group ..."
groupadd -r clamd || exit 1

echo "Adding clamd system account ..."
useradd -r -g clamd -s /bin/false -d /var/clamd clamd || exit 1

echo "Creating /var/clamd ..."
mkdir -p /var/clamd || exit 1

# on fedora core 3 libidn-devel was required

# apply clamav-0.83-stderr-and-stdout.patch
# It is a modified version of the stderr-patch from gluelogic
# (http://www.gluelogic.com/code/daemontools/clamav-0.75.1-gps-stderr.patch)
# It will allow us to do all clamd logging with multilog
echo "Applying clamav-stderr-and-stdout.patch ..."
(cd $EXTRACT_DIR/clamav-$VERSION_CLAMAV && patch -p1 < $PATCHES_DIR/clamav-stderr-and-stdout.patch) || exit 1

# note: on FC2 & FC3, configure will think zlib is outdated; therefore
# we use the option --disable-zlib-vcheck
echo "Running ./configure for clamav ..."
(cd $EXTRACT_DIR/clamav-$VERSION_CLAMAV && \
  ./configure --with-user=clamd --with-group=clamd --disable-zlib-vcheck) || exit 1

echo "Running make for clamav ..."
(cd $EXTRACT_DIR/clamav-$VERSION_CLAMAV && make) || exit 1

echo "Running make check for clamav ..."
(cd $EXTRACT_DIR/clamav-$VERSION_CLAMAV && make check) || exit 1

echo "Running make install for clamav ..."
(cd $EXTRACT_DIR/clamav-$VERSION_CLAMAV && make install) || exit 1

# test your installation
# this should find a couple of test-viruses
echo "Running clamscan on $EXTRACT_DIR/clamav-$VERSION_CLAMAV/test/ ..."
clamscan -r $EXTRACT_DIR/clamav-$VERSION_CLAMAV/test/
echo "This should have found a couple of test viruses"
echo -n "Confirm that clamscan worked by pressing ENTER:"
read in

# - comment out Example
# - change LocalSocket to /tmp/clamd.sock
# - Uncomment and set LogFile to /dev/stdout
# - Uncomment "MaxThreads" and change value to "30"
# - Uncomment "User" and change value to "clamd"
# - Uncomment "Foreground"
# - Uncomment "ScanMail"
echo "Applying patch clamd_conf.patch to /usr/local/etc/clamd.conf ..."
(cd /usr/local/etc && patch < $PATCHES_DIR/clamd_conf.patch) || exit 1

echo "Starting test instance of clamd in the background ..."
clamd &

sleep 5

# this should find a couple of test-viruses
echo "Scanning $EXTRACT_DIR/clamav-$VERSION_CLAMAV/test/ ..."
clamdscan $EXTRACT_DIR/clamav-$VERSION_CLAMAV/test/
echo -n "Confirm that the test viruses were found by pressing ENTER:"
read in

echo "Killing test instance of clamd again ..."
killall clamd
sleep 3


#############################################
# Setting up Clamd for use with daemontools #
#############################################

# resources:
# - http://www.clamav.net/doc/0.75/clamd_supervised/clamd-daemontools-guide.txt
# - http://www.downloadfreetrial.com/linux/HOWTO/Qmail-ClamAV-HOWTO/x142.html
# we already adjusted /usr/local/etc/clamd.conf
echo "NOW SETTING UP CLAMD FOR USE WITH DAEMONTOOLS ..."

echo "Creating /var/clamd/bin ..."
mkdir -p /var/clamd/bin || exit 1

echo "Installing /var/clamd/bin/clamdctl ..."
cp $SCRIPTS_DIR/var_clamd_bin_clamdctl /var/clamd/bin/clamdctl || exit

echo "Changing ownership and permissions of /var/clamd/bin/clamdctl ..."
chmod 755 /var/clamd/bin/clamdctl
chown root:root /var/clamd/bin/clamdctl

echo "Linking clamdctl into /usr/local/bin ..."
rm -f /usr/local/bin/clamdctl
ln -s /var/clamd/bin/clamdctl /usr/local/bin || exit 1

echo "Creating the supervise directories for the clamd service ..."
mkdir -p /var/clamd/supervise/clamd/log || exit 1

echo "Installing supervise scripts ..."
cp $SCRIPTS_DIR/var_clamd_supervise_clamd_run /var/clamd/supervise/clamd/run || exit 1
cp $SCRIPTS_DIR/var_clamd_supervise_clamd_log_run /var/clamd/supervise/clamd/log/run || exit 1

echo "Creating /var/clamd/control ..."
mkdir -p /var/clamd/control || exit 1

# set the memory limit of clamd
# NOTE: by default the memory limit is DEACTIVATED in /var/clamd/supervise/clamd/run
echo "Setting /var/clamd/control/clamdmemorylimit to $CLAMD_MEMORY_LIMIT ..."
echo $CLAMD_MEMORY_LIMIT > /var/clamd/control/clamdmemorylimit || exit 1

# set the maximum multilog log file size in bytes; if the logfile is
# bigger than that it will be rotated
echo "Setting /var/clamd/control/multilogsize to $CLAMD_MULTILOGSIZE ..."
echo $CLAMD_MULTILOGSIZE > /var/clamd/control/multilogsize || exit 1

# set the maximum number of multilog log files to keep; old log files will be deleted
echo "Setting /var/clamd/control/multilognum to $CLAMD_MULTILOGNUM ..."
echo $CLAMD_MULTILOGNUM > /var/clamd/control/multilognum || exit 1

# be paranoid
echo "Setting permissions for /var/clamd/control ..."
chmod 700 /var/clamd/control || exit 1
chmod 600 /var/clamd/control/* || exit 1

# Make the run files executable:
echo "Setting execute permissions for supervise scripts ..."
chmod 755 /var/clamd/supervise/clamd/run || exit 1
chmod 755 /var/clamd/supervise/clamd/log/run || exit 1

# Then set up the log directory:
echo "Creating /var/log/clamd ..."
mkdir -p /var/log/clamd || exit 1
chown qmaill:root /var/log/clamd || exit 1

# Finally, link the supervise directory into /service:
echo "Linking /var/clamd/supervise/clamd into /service ..."
ln -s /var/clamd/supervise/clamd /service || exit 1

echo "Running telinit q"
telinit q

echo "Waiting 5 seconds for clamd to start up ..."
sleep 5

echo "Running clamdctl stat ..."
clamdctl stat
echo -n "Confirm that clamd successfully started by pressing ENTER:"
read in
 
echo "Testing clamd by running clamdscan again ..."
clamdscan $EXTRACT_DIR/clamav-$VERSION_CLAMAV/test/
echo "You might want to check the log file /var/log/clamd/current."
echo "An error such as pthread_create failed would indicate that the limits are too low."
echo -n "Confirm that clamdscan found a couple of test viruses by pressing ENTER:"
read in

# If it fails with "ERROR: pthread_create failed":
#  - try to edit /var/clamd/supervise/clamd/run and run clamd without a softlimit
#  - if that works, try to replace the softlimit line with
#  exec /usr/local/bin/softlimit -d $CLAMDMEMORYLIMIT -s $CLAMDMEMORYLIMIT -l $CLAMDMEMORYLIMIT \
#  by default the -m option is used which is equivalent to using the options -l -s -d and -a; but
#  it seems that the -a option is causing the trouble.


#########################################
# setting up freshclam with daemontools #
#########################################
echo "NOW SETTING UP FRESHCLAM WITH DAEMONTOOLS:"

# comment out Example
# uncomment Foreground
# make sure UpdateLogFile is commented out - multilog will do the logging
echo "Applying patch freshclam_conf.patch ..."
(cd /usr/local/etc && patch < $PATCHES_DIR/freshclam_conf.patch) || exit 1

echo "Installing /var/clamd/bin/freshclamctl ..."
cp $SCRIPTS_DIR/var_clamd_bin_freshclamctl /var/clamd/bin/freshclamctl || exit 1

echo "Changing ownership and permissions for /var/clamd/bin/freshclamctl ..."
chmod 755 /var/clamd/bin/freshclamctl
chown root:root /var/clamd/bin/freshclamctl
echo "Linking freshclamctl into /usr/local/bin ..."
rm -f /usr/local/bin/freshclamctl
ln -s /var/clamd/bin/freshclamctl /usr/local/bin || exit 1

echo "Creating supervise directories for freshclam service ..."
mkdir -p /var/clamd/supervise/freshclam/log || exit 1

echo "Installing supervise scripts ..."
cp $SCRIPTS_DIR/var_clamd_supervise_freshclam_run \
  /var/clamd/supervise/freshclam/run || exit 1
cp $SCRIPTS_DIR/var_clamd_supervise_freshclam_log_run \
  /var/clamd/supervise/freshclam/log/run || exit 1

# how often to check for virus updates
echo "Setting /var/clamd/control/freshclam_checksperday to $FRESHCLAM_CHECKS_PER_DAY ..."
echo $FRESHCLAM_CHECKS_PER_DAY > /var/clamd/control/freshclam_checksperday || exit 1

# set the memory limit of freshclam
echo "Setting /var/clamd/control/freshclam_memorylimit to $FRESHCLAM_MEMORYLIMIT ..."
echo $FRESHCLAM_MEMORYLIMIT > /var/clamd/control/freshclam_memorylimit || exit 1

# set the maximum multilog log file size in bytes; if the logfile
# is bigger than that it will be rotated
echo "Setting /var/clamd/control/freshclam_multilogsize to $FRESHCLAM_MULTILOGSIZE ..."
echo $FRESHCLAM_MULTILOGSIZE > /var/clamd/control/freshclam_multilogsize || exit 1

# set the maximum number of multilog log files to keep; old log files will be deleted
echo "Setting /var/clamd/control/freshclam_multilognum to $FRESHCLAM_MULTILOGNUM ..."
echo $FRESHCLAM_MULTILOGNUM > /var/clamd/control/freshclam_multilognum || exit 1

# Make the run files executable:
echo "Setting execute permissions for supervise scripts ..."
chmod 755 /var/clamd/supervise/freshclam/run || exit 1
chmod 755 /var/clamd/supervise/freshclam/log/run || exit 1

echo "Creating /var/log/freshclam ..."
mkdir -p /var/log/freshclam || exit 1
chown qmaill:root /var/log/freshclam || exit 1

echo "Linking the supervise directory into /service ..."
ln -s /var/clamd/supervise/freshclam /service || exit 1

echo "Running telinit q"
telinit q

echo "Waiting 5 seconds for freshclam to startup ..."
sleep 5

echo "Showing last ten lines of /var/log/freshclam/current ..."
tail /var/log/freshclam/current
echo "Running freshclamctl stat ..."
freshclamctl stat
echo -n "Confirm that freshclam successfully started by pressing ENTER:"
read in

#################
# QMAIL-QFILTER #
#################
# qmail-qfilter was chosen over qmail-scanner for the following reasons:
# - it is much faster, as it does not need to load the perl interpreter
# - it has less features than qmail-scanner but the most useful one of
#   denying messages with certain attachments can be easy implemented
#   with qmail-qfilter if needed
echo "INSTALLING QMAIL-QFILTER:"

echo "Extracting bglibs ..."
(cd $EXTRACT_DIR && tar zxf $PACKAGES_DIR/bglibs-$VERSION_BGLIBS.tar.gz) || exit 1

echo "Creating /usr/local/bglibs/ ..."
mkdir /usr/local/bglibs/

echo "Running make for bglibs ..."
(cd $EXTRACT_DIR/bglibs-$VERSION_BGLIBS && make) || exit 1

echo "Running make install for bglibs ..."
(cd $EXTRACT_DIR/bglibs-$VERSION_BGLIBS && make install) || exit 1

echo "Extracting qmail-qfilter ..."
(cd $EXTRACT_DIR && tar zxf $PACKAGES_DIR/qmail-qfilter-$VERSION_QMAIL_QFILTER.tar.gz) || exit 1

# changing TMPDIR; it gets set to a temporary directory that
# only the executor of qmail-qfilter (qmaild) has write access to;
# we use /usr/local/qmail-qfilter-tmp
echo "Applying patch qmail-qfilter_TMP.patch ..."
(cd $EXTRACT_DIR/qmail-qfilter-$VERSION_QMAIL_QFILTER && patch < $PATCHES_DIR/qmail-qfilter_TMPDIR.patch) || exit 1

echo "Creating qmail-qfilter TMPDIR /usr/local/qmail-qfilter-tmp ..."
mkdir -p /usr/local/qmail-qfilter-tmp || exit 1

# only qmaild (which runs qmail-qfilter.sh-->qmail-qfilter-->qmail-qfilter-clamdscan.sh) will
# be able to write to the temporary files but clamd (which runs clamd) will be able to read
# them to perform the virus checking.
echo "Setting ownership and permissions for /usr/local/qmail-qfilter-tmp ..."
chown qmaild:clamd /usr/local/qmail-qfilter-tmp || exit 1
chmod 750 /usr/local/qmail-qfilter-tmp || exit 1

# build it and install in /usr/local/bin/
echo "Running make for qmail-qfilter ..."
(cd $EXTRACT_DIR/qmail-qfilter-$VERSION_QMAIL_QFILTER && make) || exit 1

echo "Running make install for qmail-qfilter ..."
(cd $EXTRACT_DIR/qmail-qfilter-$VERSION_QMAIL_QFILTER && make install) || exit 1

echo "Installing /usr/local/bin/qmail-qfilter.sh ..."
cp $SCRIPTS_DIR/usr_local_bin_qmail-qfilter.sh \
  /usr/local/bin/qmail-qfilter.sh || exit 1

echo "Installing /usr/local/bin/qmail-qfilter-clamdscan.sh ..."
cp $SCRIPTS_DIR/usr_local_bin_qmail-qfilter-clamdscan.sh \
  /usr/local/bin/qmail-qfilter-clamdscan.sh || exit 1

# make scripts executable
echo "Setting execution permissions for qmail-qfilter.sh & qmail-qfilter-clamdscan.sh ..."
chmod 755 /usr/local/bin/qmail-qfilter.sh || exit 1
chmod 755 /usr/local/bin/qmail-qfilter-clamdscan.sh || exit 1

# if something goes *really* wrong qmail-qfilter-clamdscan.sh logs the error to
# /var/log/qmail-qfilter-clamdscan.err; as the script runs as qmaild, the file must
# be writable by that user
echo "Creating log file /var/log/qmail-qfilter-clamdscan.err ..."
touch /var/log/qmail-qfilter-clamdscan.err || exit 1
chown qmaild:root /var/log/qmail-qfilter-clamdscan.err || exit 1

echo "Installing /etc/tcp.smtp ..."
cp $SCRIPTS_DIR/etc_tcp.smtp /etc/tcp.smtp || exit 1

echo "Running qmailctl cdb"
qmailctl cdb

echo ""
echo "CONGRATULATIONS! The second part of your qmail installation was successfully completed."
echo "Try sending regular mails now. If that works try to send"
echo "the test virus $EXTRACT_DIR/clamav-$VERSION_CLAMAV/test/clam.zip as an attachment."
echo "Meanwhile keep an eye on the logs by running somthing like"
echo "tail -fn0 /var/log/qmail/current /var/log/clamd/current"


#END

