#!/bin/bash

export HOME_DIR=`(cd $(dirname $0); pwd)`
. $HOME_DIR/common.sh || exit 1

printUsage()
{
cat >&1 <<END
Usage:
 $0 [-C] [-U]
 -C  remove extraced files
 -U  uninstall *everything*; know what you are doing!
END
}

while getopts ":CU" opt; do
    case $opt in
        C ) cleanup=true;;
        U ) uninstall=true;;
        ? ) printUsage
            exit 1 ;;
    esac
done


if [ "$cleanup" = "true" ] ; then
    echo "Removing $EXTRACT_DIR/* ..."
    rm -rf $EXTRACT_DIR/*
fi

if [ "$uninstall" = "true" ] ; then
    echo "Stopping authdaemond ..."
    /etc/init.d/authdaemond stop
    sleep 2
    killall authdaemond
    echo "Removing mailusers group and mailusers account ..."
    userdel -r mailusers
    groupdel mailusers
    echo "Removing /usr/local/etc/authlib ..."
    rm -rf /usr/local/etc/authlib
    echo "Removing files from /usr/local/sbin ..."
    rm -f /usr/local/sbin/authdaemond
    rm -f /usr/local/sbin/courierlogger
    rm -f /usr/local/sbin/authconfig
    rm -f /usr/local/sbin/authtest
    rm -f /usr/local/sbin/authenumerate
    echo "Removing /usr/local/lib/courier-authlib ..."
    rm -rf /usr/local/lib/courier-authlib
    echo "Removing /usr/local/libexec/courier-authlib ..."
    rm -rf /usr/local/libexec/courier-authlib
    echo "Removing /usr/local/var/spool/authdaemon ..."
    rm -rf /usr/local/var/spool/authdaemon
    echo "Reverting /var/qmail/supervise/qmail-smtpd/run to old state..."
    cp $SCRIPTS_DIR/var_qmail_supervise_qmail-smtpd_run \
      /var/qmail/supervise/qmail-smtpd/run || exit 1
fi

if [ "$uninstall" = "true" -o "$cleanup" = "true" ] ; then
    exit 0
fi

### PREREQUISITES ###
echo "Checking prerequisites:"
checkRPM expect
checkRPM libtool
checkRPMAlternative gdbm-devel db-devel

if [ "$MYSQL_ROOT_PWD" = "" ] ; then
    echo "ERROR: Edit install.conf and set MYSQL_ROOT_PWD!"
    exit 1
fi

if [ "$MYSQL_MAILADMIN_PWD" = "" ] ; then
    echo "ERROR: Edit install.conf and set MYSQL_MAILADMIN_PWD!"
    exit 1
fi

if [ "$MYSQL_MAILUSERS_PWD" = "" ] ; then
    echo "ERROR: Edit install.conf and set MYSQL_MAILUSERS_PWD!"
    exit 1
fi

mkdir -p $PACKAGES_DIR || exit 1
mkdir -p $EXTRACT_DIR || exit 1


###################
# COURIER AUTHLIB #
###################
# install the Courier authentication library

echo "Extracting courier-authlib ..."
(cd $EXTRACT_DIR && tar jxf $PACKAGES_DIR/courier-authlib-$VERSION_AUTHLIB.tar.bz2) || exit 1

echo "Creating system group mailusers ..."
groupadd mailusers || exit 1
echo "Creating system account mailusers ..."
useradd -g mailusers -d /home/mailusers -s /bin/false mailusers || exit 1

# for the authentication modules see http://www.courier-mta.org/authlib/README_authlib.html
# we include --with-authuserdb so that userdbpw gets built; we will need it for password ecryption
# --with-mailuser and --with-mailgroup specify the user/group that will own the configuration files
# and the socket that authentication daemon process listens on
# include the --with-redhat option when installing on fedora
echo "Running ./configure for courier-authlib ..."
(cd $EXTRACT_DIR/courier-authlib-$VERSION_AUTHLIB &&
./configure \
--with-mailuser=mailusers \
--with-mailgroup=mailusers \
--with-authshadow \
--with-authmysql \
--with-authuserdb \
--without-authsasl \
--without-authpwd \
--without-authpam \
--without-authvchkpw \
--without-authpgsql \
--without-authldap \
--without-authcustom \
--with-mysql-libs=/usr/local/mysql/lib/mysql \
--with-mysql-includes=/usr/local/mysql/include/mysql) || exit 1

echo "Running make for courier-authlib ..."
(cd $EXTRACT_DIR/courier-authlib-$VERSION_AUTHLIB && make) || exit 1

echo "Running make install for courier-authlib ..."
(cd $EXTRACT_DIR/courier-authlib-$VERSION_AUTHLIB && make install) || exit 1

echo "Running make install-migrate for courier-authlib ..."
(cd $EXTRACT_DIR/courier-authlib-$VERSION_AUTHLIB && make install-migrate) || exit 1

echo "Running make install-configure for courier-authlib ..."
(cd $EXTRACT_DIR/courier-authlib-$VERSION_AUTHLIB && make install-configure) || exit 1

echo "Running make install-man for courier-authlib ..."
(cd $EXTRACT_DIR/courier-authlib-$VERSION_AUTHLIB && make install-man) || exit 1

# set authmodulelist="authshadow authmysql"
echo "Applying patch authdaemonrc.patch ..."
(cd /usr/local/etc/authlib && patch < $PATCHES_DIR/authdaemonrc.patch) || exit 1

# What gets installed:
#  - /usr/local/etc/authlib - the configuration files.
#  - /usr/local/sbin - the authdaemond startup script; several utility
#    programs (courierlogger, authconfig, authtest, authenumerate); and userdb scripts.
#  - /usr/local/lib/courier-authlib - various authentication modules, as shared libraries.
#  - /usr/local/libexec/courier-authlib - some miscellaneous stuff.
#  - /usr/local/var/spool/authdaemon - a subdirectory that contains the
#    filesystem socket which authdaemond listens on.
#  - /usr/local/include - a header file that Courier packages will use to
#    build against courier-authlib.

echo "Installing /etc/init.d/authdaemond ..."
cp $EXTRACT_DIR/courier-authlib-$VERSION_AUTHLIB/courier-authlib.sysvinit /etc/init.d/authdaemond || exit 1

echo "Setting execute permissions on /etc/init.d/authdaemond ..."
chmod 755 /etc/init.d/authdaemond || exit 1


# should be started after mysql and before courier-imap & courier-pop
echo "Creating run level startup scripts ..."
rm -f /etc/rc.d/rc3.d/S13authdaemond
ln -s /etc/init.d/authdaemond /etc/rc.d/rc3.d/S13authdaemond || exit 1
# should be killed before mysql and after courier-imap & courier-pop
rm -f /etc/rc.d/rc3.d/K07authdaemond
ln -s /etc/init.d/authdaemond /etc/rc.d/rc3.d/K07authdaemond || exit 1

# run manually to see if all auth modules get loaded successfully
echo "We'll run authdaemond now ..."
/usr/local/libexec/courier-authlib/authdaemond &
sleep 3
killall authdaemond 2>/dev/null 1>&1
echo -n "Confirm that all modules loaded successfully by pressing ENTER:"
read in

echo "Changing permissions so that only the user mailusers (& root) can talk to the authdaemon ..."
chmod 700 /usr/local/var/spool/authdaemon || exit 1

# as authdaemon runs as root, root and not mailusers should own these files
echo "Changing ownership and permissions for /usr/local/etc/authlib/* ..."
chown root:root /usr/local/etc/authlib/* || exit 1
chmod 600 /usr/local/etc/authlib/* || exit 1

echo "Starting authdaemond ..."
/etc/init.d/authdaemond start || exit 1
sleep 3

echo "Running 'authtest root' ..."
authtest root || exit 1

echo "Running authenumerate (this should show all local accounts) ..."
authenumerate
echo -n "Confirm that it works by pressing ENTER:"
read in

echo "Test authlib and authdaemond by runing 'authtest <user> <password>' on another console."
echo -n "Confirm that it works by pressing ENTER:"
read in


# configure authmysql
# - comment out MYSQL_SERVER and MYSQL_PORT and set MYSQL_SOCKET to /tmp/mysql.sock
# - set MYSQL_USERNAME to "mailusers" and MYSQL_PASSWORD to what you have chosen before
#   mailusers has only SELECT privileges in the database mailusers
# - set MYSQL_DATABASE to "mailusers"
# - uncomment MYSQL_QUOTA_FIELD
echo "Applying patch authmysqlrc.patch ..."
sed s/MYSQL_MAILUSERS_PWD/$MYSQL_MAILUSERS_PWD/ $PATCHES_DIR/authmysqlrc.patch | (cd /usr/local/etc/authlib && patch) || exit 1

echo "Overwriting /var/qmail/supervise/qmail-smtpd/run ..."
cp $SCRIPTS_DIR/var_qmail_supervise_qmail-smtpd_run_COURIERPASSWD \
  /var/qmail/supervise/qmail-smtpd/run || exit 1

echo ""
echo "CONGRATULATIONS! authlib & authdaemond were successfully installed."

# END
